/**
 * Notebook Clipper - Service Worker
 * 백그라운드에서 실행되는 서비스 워커
 */

const API_BASE = 'https://notebook.sangki.com';

// 확장프로그램 설치 시
chrome.runtime.onInstalled.addListener((details) => {
    console.log('Notebook Clipper installed:', details.reason);

    // 기존 컨텍스트 메뉴 제거 후 생성
    chrome.contextMenus.removeAll(() => {
        chrome.contextMenus.create({
            id: 'save-to-notebook',
            title: 'Notebook에 저장',
            contexts: ['page']
        });

        chrome.contextMenus.create({
            id: 'save-selection-to-notebook',
            title: '선택 영역 Notebook에 저장',
            contexts: ['selection']
        });
    });
});

// 컨텍스트 메뉴 클릭 이벤트
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
    if (info.menuItemId === 'save-to-notebook' || info.menuItemId === 'save-selection-to-notebook') {
        const selectionOnly = info.menuItemId === 'save-selection-to-notebook';

        // 인증 확인
        const stored = await chrome.storage.local.get(['token', 'user']);
        if (!stored.token) {
            // 팝업 열기 (알림으로 대체)
            chrome.notifications.create({
                type: 'basic',
                iconUrl: 'icons/icon128.png',
                title: '로그인 필요',
                message: '확장프로그램 아이콘을 클릭하여 먼저 로그인해주세요.'
            });
            return;
        }

        // 저장 실행
        try {
            await saveCurrentPage(tab, stored.token, selectionOnly);
            chrome.notifications.create({
                type: 'basic',
                iconUrl: 'icons/icon128.png',
                title: '저장 완료',
                message: '노트가 Notebook에 저장되었습니다.'
            });
        } catch (error) {
            chrome.notifications.create({
                type: 'basic',
                iconUrl: 'icons/icon128.png',
                title: '저장 실패',
                message: error.message || '저장 중 오류가 발생했습니다.'
            });
        }
    }
});

// 메시지 리스너
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'checkAuth') {
        checkAuth().then(sendResponse);
        return true;
    }

    if (request.action === 'logout') {
        logout().then(sendResponse);
        return true;
    }

    return false;
});

/**
 * 인증 상태 확인
 */
async function checkAuth() {
    const stored = await chrome.storage.local.get(['token', 'user']);

    if (!stored.token) {
        return { authenticated: false };
    }

    try {
        const response = await fetch(`${API_BASE}/account/api/verify/`, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${stored.token}`
            }
        });

        if (response.ok) {
            return {
                authenticated: true,
                user: stored.user
            };
        } else {
            await chrome.storage.local.remove(['token', 'user']);
            return { authenticated: false };
        }
    } catch (error) {
        console.error('Auth check failed:', error);
        return { authenticated: false };
    }
}

/**
 * 로그아웃
 */
async function logout() {
    const stored = await chrome.storage.local.get(['token']);

    if (stored.token) {
        try {
            await fetch(`${API_BASE}/account/api/logout/`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${stored.token}`
                }
            });
        } catch (error) {
            console.error('Logout request failed:', error);
        }
    }

    await chrome.storage.local.remove(['token', 'user']);
    return { success: true };
}

/**
 * 현재 페이지 저장
 */
async function saveCurrentPage(tab, token, selectionOnly = false) {
    // Content script 실행하여 콘텐츠 추출
    const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: extractContent,
        args: [true, selectionOnly]
    });

    if (!results || !results[0] || !results[0].result) {
        throw new Error('페이지 콘텐츠를 추출할 수 없습니다.');
    }

    const { markdown, images } = results[0].result;

    // 노트 생성
    const noteResponse = await fetch(`${API_BASE}/notes/api/notes/`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
            title: tab.title || '제목 없음',
            content: markdown,
            source_url: tab.url
        })
    });

    if (!noteResponse.ok) {
        const error = await noteResponse.json();
        throw new Error(error.error || '노트 생성에 실패했습니다.');
    }

    const noteData = await noteResponse.json();

    // 이미지 업로드 (최대 5개)
    if (images && images.length > 0) {
        for (const imageUrl of images.slice(0, 5)) {
            try {
                const imgResponse = await fetch(imageUrl);
                if (!imgResponse.ok) continue;

                const blob = await imgResponse.blob();
                const filename = imageUrl.split('/').pop().split('?')[0] || 'image.jpg';

                const formData = new FormData();
                formData.append('files', blob, filename);

                await fetch(`${API_BASE}/notes/api/notes/${noteData.id}/attachments/`, {
                    method: 'POST',
                    headers: {
                        'Authorization': `Bearer ${token}`
                    },
                    body: formData
                });
            } catch (e) {
                console.error('Image upload failed:', e);
            }
        }
    }

    return noteData;
}

/**
 * 페이지 콘텐츠 추출 함수 (injected)
 */
function extractContent(includeImages, selectionOnly) {
    function htmlToMarkdown(element) {
        function processNode(node) {
            if (node.nodeType === Node.TEXT_NODE) {
                return node.textContent;
            }
            if (node.nodeType !== Node.ELEMENT_NODE) {
                return '';
            }

            const tag = node.tagName.toLowerCase();
            let content = '';
            for (const child of node.childNodes) {
                content += processNode(child);
            }
            content = content.trim();

            switch (tag) {
                case 'h1': return `# ${content}\n\n`;
                case 'h2': return `## ${content}\n\n`;
                case 'h3': return `### ${content}\n\n`;
                case 'h4': return `#### ${content}\n\n`;
                case 'p': return content ? `${content}\n\n` : '';
                case 'br': return '\n';
                case 'hr': return '\n---\n\n';
                case 'strong':
                case 'b': return content ? `**${content}**` : '';
                case 'em':
                case 'i': return content ? `*${content}*` : '';
                case 'code': return content ? `\`${content}\`` : '';
                case 'pre': return `\n\`\`\`\n${content}\n\`\`\`\n\n`;
                case 'blockquote': return content ? `> ${content}\n\n` : '';
                case 'a': {
                    const href = node.getAttribute('href') || '';
                    return content ? `[${content}](${href})` : '';
                }
                case 'img': {
                    if (!includeImages) return '';
                    const src = node.getAttribute('src') || '';
                    const alt = node.getAttribute('alt') || '';
                    return src ? `![${alt}](${src})\n` : '';
                }
                case 'ul':
                case 'ol': return `${content}\n`;
                case 'li': {
                    const parent = node.parentElement;
                    if (parent && parent.tagName.toLowerCase() === 'ol') {
                        const index = Array.from(parent.children).indexOf(node) + 1;
                        return `${index}. ${content}\n`;
                    }
                    return `- ${content}\n`;
                }
                case 'script':
                case 'style':
                case 'nav':
                case 'footer':
                case 'header':
                case 'aside':
                    return '';
                default:
                    return content;
            }
        }
        return processNode(element).trim();
    }

    if (selectionOnly) {
        const selection = window.getSelection();
        if (selection && selection.rangeCount > 0 && selection.toString().trim()) {
            const range = selection.getRangeAt(0);
            const container = document.createElement('div');
            container.appendChild(range.cloneContents());
            return { markdown: htmlToMarkdown(container), images: [] };
        }
    }

    const selectors = ['article', '[role="main"]', 'main', '.content', '.post'];
    let mainContent = null;

    for (const selector of selectors) {
        mainContent = document.querySelector(selector);
        if (mainContent) break;
    }

    if (!mainContent) {
        mainContent = document.body.cloneNode(true);
        ['script', 'style', 'nav', 'footer', 'header', 'aside'].forEach(sel => {
            mainContent.querySelectorAll(sel).forEach(el => el.remove());
        });
    }

    const markdown = htmlToMarkdown(mainContent);
    const images = [];

    if (includeImages) {
        mainContent.querySelectorAll('img').forEach(img => {
            const src = img.getAttribute('src');
            if (src && src.startsWith('http') && !images.includes(src)) {
                images.push(src);
            }
        });
    }

    return { markdown, images };
}
