/**
 * Notebook Clipper - Content Script
 * 웹 페이지에 주입되어 콘텐츠 추출을 담당
 * HTML을 추출하여 서버로 전송 (서버에서 정제)
 */

// 메시지 리스너
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'extractContent') {
        const result = extractPageContent(
            request.includeImages ?? true,
            request.selectionOnly ?? false
        );
        sendResponse(result);
    }

    if (request.action === 'getSelection') {
        const selection = window.getSelection();
        sendResponse({
            hasSelection: selection && selection.toString().length > 0,
            text: selection ? selection.toString() : ''
        });
    }

    return true;
});

/**
 * 페이지 콘텐츠 추출
 * HTML + CSS 원본을 추출하여 반환 (서버에서 BeautifulSoup으로 정제)
 */
function extractPageContent(includeImages = true, selectionOnly = false) {
    const baseUrl = window.location.href;
    const origin = window.location.origin;

    // URL을 절대 경로로 변환하는 헬퍼 함수
    function toAbsoluteUrl(url) {
        if (!url || url.startsWith('data:') || url.startsWith('javascript:') || url.startsWith('#')) {
            return url;
        }
        if (url.startsWith('http://') || url.startsWith('https://')) {
            return url;
        }
        try {
            return new URL(url, baseUrl).href;
        } catch (e) {
            return url;
        }
    }

    // 선택 영역만 저장하는 경우
    if (selectionOnly) {
        const selection = window.getSelection();
        if (selection && selection.rangeCount > 0 && selection.toString().trim()) {
            const range = selection.getRangeAt(0);
            const container = document.createElement('div');
            container.appendChild(range.cloneContents());

            // 선택 영역 내 링크와 이미지도 절대 경로로 변환
            container.querySelectorAll('a[href]').forEach(a => {
                a.href = toAbsoluteUrl(a.getAttribute('href'));
            });
            container.querySelectorAll('img[src]').forEach(img => {
                img.src = toAbsoluteUrl(img.getAttribute('src'));
            });

            return { html: container.innerHTML, images: [] };
        }
    }

    // body 복제하여 작업 (원본 수정 방지)
    const bodyClone = document.body.cloneNode(true);

    // 모든 링크를 절대 경로로 변환
    bodyClone.querySelectorAll('a[href]').forEach(a => {
        const href = a.getAttribute('href');
        a.setAttribute('href', toAbsoluteUrl(href));
    });

    // 모든 이미지 src를 절대 경로로 변환
    bodyClone.querySelectorAll('img').forEach(img => {
        const src = img.getAttribute('src');
        const dataSrc = img.getAttribute('data-src');
        const dataLazySrc = img.getAttribute('data-lazy-src');

        if (src) img.setAttribute('src', toAbsoluteUrl(src));
        if (dataSrc) img.setAttribute('data-src', toAbsoluteUrl(dataSrc));
        if (dataLazySrc) img.setAttribute('data-lazy-src', toAbsoluteUrl(dataLazySrc));
    });

    // srcset 속성도 처리
    bodyClone.querySelectorAll('[srcset]').forEach(el => {
        const srcset = el.getAttribute('srcset');
        if (srcset) {
            const newSrcset = srcset.split(',').map(part => {
                const [url, descriptor] = part.trim().split(/\s+/);
                return descriptor ? `${toAbsoluteUrl(url)} ${descriptor}` : toAbsoluteUrl(url);
            }).join(', ');
            el.setAttribute('srcset', newSrcset);
        }
    });

    // CSS 추출 (style 태그 + link 태그의 인라인 스타일)
    let cssContent = '';

    // 1. <style> 태그 내용 수집
    document.querySelectorAll('style').forEach(style => {
        cssContent += style.textContent + '\n';
    });

    // 2. 외부 CSS는 URL만 수집 (link 태그)
    const cssLinks = [];
    document.querySelectorAll('link[rel="stylesheet"]').forEach(link => {
        if (link.href) {
            cssLinks.push(link.href);
        }
    });

    // 이미지 URL 추출 (절대 경로로)
    const images = [];
    if (includeImages) {
        bodyClone.querySelectorAll('img').forEach(img => {
            let src = img.getAttribute('src') || img.getAttribute('data-src') || img.getAttribute('data-lazy-src');
            if (src && !src.startsWith('data:') && !images.includes(src)) {
                images.push(src);
            }
        });
    }

    // HTML 원본 + CSS 함께 전송
    const fullHtml = `<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<base href="${origin}">
${cssLinks.map(url => `<link rel="stylesheet" href="${url}">`).join('\n')}
<style>${cssContent}</style>
</head>
<body>
${bodyClone.innerHTML}
</body>
</html>`;

    return { html: fullHtml, images };
}

// 페이지 로드 완료 알림
console.log('Notebook Clipper content script loaded');
