/**
 * Notebook Clipper - API 통신 모듈
 * notebook.sangki.com API와 통신
 */

const NotebookAPI = {
    BASE_URL: 'https://notebook.sangki.com',

    /**
     * 로그인
     * @param {string} email
     * @param {string} password
     * @returns {Promise<{success: boolean, token?: string, user?: object, error?: string}>}
     */
    async login(email, password) {
        try {
            const response = await fetch(`${this.BASE_URL}/account/api/login/`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ email, password })
            });

            const data = await response.json();

            if (response.ok) {
                return {
                    success: true,
                    token: data.token,
                    user: data.user,
                    expiresAt: data.expires_at
                };
            } else {
                return {
                    success: false,
                    error: data.error || '로그인에 실패했습니다.'
                };
            }
        } catch (error) {
            console.error('Login API error:', error);
            return {
                success: false,
                error: '서버 연결에 실패했습니다.'
            };
        }
    },

    /**
     * 토큰 검증
     * @param {string} token
     * @returns {Promise<boolean>}
     */
    async verifyToken(token) {
        try {
            const response = await fetch(`${this.BASE_URL}/account/api/verify/`, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            return response.ok;
        } catch (error) {
            console.error('Verify token error:', error);
            return false;
        }
    },

    /**
     * 로그아웃
     * @param {string} token
     * @returns {Promise<boolean>}
     */
    async logout(token) {
        try {
            await fetch(`${this.BASE_URL}/account/api/logout/`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            return true;
        } catch (error) {
            console.error('Logout API error:', error);
            return false;
        }
    },

    /**
     * 노트 생성
     * @param {string} token
     * @param {object} noteData - { title, content, source_url }
     * @returns {Promise<{success: boolean, note?: object, error?: string}>}
     */
    async createNote(token, noteData) {
        try {
            const response = await fetch(`${this.BASE_URL}/notes/api/notes/`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify(noteData)
            });

            const data = await response.json();

            if (response.ok) {
                return {
                    success: true,
                    note: data
                };
            } else {
                return {
                    success: false,
                    error: data.error || '노트 생성에 실패했습니다.'
                };
            }
        } catch (error) {
            console.error('Create note API error:', error);
            return {
                success: false,
                error: '서버 연결에 실패했습니다.'
            };
        }
    },

    /**
     * 첨부파일 업로드
     * @param {string} token
     * @param {string} noteId
     * @param {Blob} file
     * @param {string} filename
     * @returns {Promise<{success: boolean, attachment?: object, error?: string}>}
     */
    async uploadAttachment(token, noteId, file, filename) {
        try {
            const formData = new FormData();
            formData.append('files', file, filename);

            const response = await fetch(`${this.BASE_URL}/notes/api/notes/${noteId}/attachments/`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`
                },
                body: formData
            });

            const data = await response.json();

            if (response.ok) {
                return {
                    success: true,
                    attachments: data.uploaded
                };
            } else {
                return {
                    success: false,
                    error: data.error || '파일 업로드에 실패했습니다.'
                };
            }
        } catch (error) {
            console.error('Upload attachment API error:', error);
            return {
                success: false,
                error: '파일 업로드에 실패했습니다.'
            };
        }
    },

    /**
     * 폴더 목록 조회
     * @param {string} token
     * @returns {Promise<{success: boolean, folders?: array, error?: string}>}
     */
    async getFolders(token) {
        try {
            const response = await fetch(`${this.BASE_URL}/notes/api/folders/`, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });

            const data = await response.json();

            if (response.ok) {
                return {
                    success: true,
                    folders: data.folders,
                    rootNotes: data.root_notes
                };
            } else {
                return {
                    success: false,
                    error: data.error || '폴더 목록을 가져올 수 없습니다.'
                };
            }
        } catch (error) {
            console.error('Get folders API error:', error);
            return {
                success: false,
                error: '서버 연결에 실패했습니다.'
            };
        }
    }
};

// 전역으로 내보내기
if (typeof window !== 'undefined') {
    window.NotebookAPI = NotebookAPI;
}
