/**
 * Notebook Clipper - Popup Script
 * 웹 페이지를 Notebook에 저장하는 Chrome 확장프로그램
 */

// DOM Elements
const elements = {
    loginSection: document.getElementById('login-section'),
    saveSection: document.getElementById('save-section'),
    loginForm: document.getElementById('login-form'),
    loginError: document.getElementById('login-error'),
    loginBtn: document.getElementById('login-btn'),
    logoutBtn: document.getElementById('logout-btn'),
    saveBtn: document.getElementById('save-btn'),
    saveStatus: document.getElementById('save-status'),
    userName: document.getElementById('user-name'),
    pageTitle: document.getElementById('page-title'),
    pageUrl: document.getElementById('page-url'),
    includeImages: document.getElementById('include-images'),
    saveSelection: document.getElementById('save-selection'),
    loadingOverlay: document.getElementById('loading-overlay'),
    loadingText: document.getElementById('loading-text')
};

// State
let state = {
    token: null,
    user: null,
    currentTab: null
};

/**
 * 초기화
 */
async function init() {
    try {
        // 저장된 인증 정보 확인
        const stored = await chrome.storage.local.get(['token', 'user']);

        if (stored.token && stored.user) {
            // 토큰 유효성 검증
            showLoading('인증 확인 중...');
            const isValid = await NotebookAPI.verifyToken(stored.token);

            if (isValid) {
                state.token = stored.token;
                state.user = stored.user;
                showSaveSection();
                await loadCurrentPage();
            } else {
                // 토큰이 유효하지 않으면 삭제
                await chrome.storage.local.remove(['token', 'user']);
                showLoginSection();
            }
            hideLoading();
        } else {
            showLoginSection();
        }
    } catch (error) {
        console.error('Init error:', error);
        showLoginSection();
        hideLoading();
    }
}

/**
 * 로그인 섹션 표시
 */
function showLoginSection() {
    elements.loginSection.classList.remove('hidden');
    elements.saveSection.classList.add('hidden');
    elements.loginError.textContent = '';
}

/**
 * 저장 섹션 표시
 */
function showSaveSection() {
    elements.loginSection.classList.add('hidden');
    elements.saveSection.classList.remove('hidden');
    elements.userName.textContent = state.user?.name || state.user?.email || '사용자';
}

/**
 * 현재 페이지 정보 로드
 */
async function loadCurrentPage() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab) {
            state.currentTab = tab;
            elements.pageTitle.value = tab.title || '제목 없음';
            elements.pageUrl.textContent = tab.url;

            // 선택된 텍스트가 있는지 확인
            try {
                const results = await chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    func: () => window.getSelection().toString()
                });
                if (results && results[0] && results[0].result) {
                    elements.saveSelection.parentElement.style.display = 'flex';
                } else {
                    elements.saveSelection.parentElement.style.display = 'none';
                    elements.saveSelection.checked = false;
                }
            } catch (e) {
                elements.saveSelection.parentElement.style.display = 'none';
            }
        }
    } catch (error) {
        console.error('Failed to load page info:', error);
    }
}

/**
 * 로딩 표시
 */
function showLoading(text = '처리 중...') {
    elements.loadingText.textContent = text;
    elements.loadingOverlay.classList.remove('hidden');
}

/**
 * 로딩 숨김
 */
function hideLoading() {
    elements.loadingOverlay.classList.add('hidden');
}

/**
 * 버튼 로딩 상태
 */
function setButtonLoading(button, loading) {
    const textEl = button.querySelector('.btn-text');
    const loadingEl = button.querySelector('.btn-loading');

    if (loading) {
        textEl?.classList.add('hidden');
        loadingEl?.classList.remove('hidden');
        button.disabled = true;
    } else {
        textEl?.classList.remove('hidden');
        loadingEl?.classList.add('hidden');
        button.disabled = false;
    }
}

/**
 * 상태 메시지 표시
 */
function showStatus(type, message) {
    elements.saveStatus.textContent = message;
    elements.saveStatus.className = `status-message ${type}`;

    if (type === 'success') {
        setTimeout(() => {
            elements.saveStatus.textContent = '';
            elements.saveStatus.className = 'status-message';
        }, 3000);
    }
}

/**
 * 로그인 처리
 */
async function handleLogin(e) {
    e.preventDefault();

    const email = document.getElementById('email').value.trim();
    const password = document.getElementById('password').value;

    if (!email || !password) {
        elements.loginError.textContent = '이메일과 비밀번호를 입력하세요.';
        return;
    }

    setButtonLoading(elements.loginBtn, true);
    elements.loginError.textContent = '';

    try {
        const result = await NotebookAPI.login(email, password);

        if (result.success) {
            // 토큰 저장
            await chrome.storage.local.set({
                token: result.token,
                user: result.user
            });

            state.token = result.token;
            state.user = result.user;

            showSaveSection();
            await loadCurrentPage();
        } else {
            elements.loginError.textContent = result.error || '로그인에 실패했습니다.';
        }
    } catch (error) {
        console.error('Login error:', error);
        elements.loginError.textContent = '서버 연결에 실패했습니다.';
    } finally {
        setButtonLoading(elements.loginBtn, false);
    }
}

/**
 * 로그아웃 처리
 */
async function handleLogout() {
    showLoading('로그아웃 중...');

    try {
        if (state.token) {
            await NotebookAPI.logout(state.token);
        }
    } catch (error) {
        console.error('Logout error:', error);
    }

    // 로컬 스토리지 정리
    await chrome.storage.local.remove(['token', 'user']);
    state.token = null;
    state.user = null;

    hideLoading();
    showLoginSection();

    // 폼 초기화
    document.getElementById('email').value = '';
    document.getElementById('password').value = '';
}

/**
 * 페이지 저장 처리
 */
async function handleSave() {
    if (!state.currentTab) {
        showStatus('error', '페이지 정보를 가져올 수 없습니다.');
        return;
    }

    const title = elements.pageTitle.value.trim() || '제목 없음';
    const includeImages = elements.includeImages.checked;
    const saveSelectionOnly = elements.saveSelection.checked;

    setButtonLoading(elements.saveBtn, true);
    elements.saveStatus.textContent = '';

    try {
        showLoading('페이지 콘텐츠 추출 중...');

        // Content script를 통해 페이지 콘텐츠 추출
        const results = await chrome.scripting.executeScript({
            target: { tabId: state.currentTab.id },
            func: extractPageContent,
            args: [includeImages, saveSelectionOnly]
        });

        if (!results || !results[0] || !results[0].result) {
            throw new Error('페이지 콘텐츠를 추출할 수 없습니다.');
        }

        const { html, images } = results[0].result;

        showLoading('노트 생성 중...');

        // 노트 생성 (HTML을 서버로 전송, 서버에서 정제)
        const noteResult = await NotebookAPI.createNote(state.token, {
            title: title,
            html: html,
            source_url: state.currentTab.url
        });

        if (!noteResult.success) {
            throw new Error(noteResult.error || '노트 생성에 실패했습니다.');
        }

        // 이미지 업로드 (선택사항)
        if (includeImages && images && images.length > 0) {
            showLoading(`이미지 업로드 중... (0/${images.length})`);
            await uploadImages(noteResult.note.id, images);
        }

        hideLoading();
        showStatus('success', '저장되었습니다! ✓');

    } catch (error) {
        console.error('Save error:', error);
        hideLoading();
        showStatus('error', error.message || '저장에 실패했습니다.');
    } finally {
        setButtonLoading(elements.saveBtn, false);
    }
}

/**
 * 이미지 업로드
 */
async function uploadImages(noteId, imageUrls) {
    const maxImages = 10;
    const imagesToUpload = imageUrls.slice(0, maxImages);
    let uploaded = 0;

    for (const url of imagesToUpload) {
        try {
            // 이미지 URL을 Blob으로 변환
            const response = await fetch(url);
            if (!response.ok) continue;

            const blob = await response.blob();
            const filename = extractFilename(url) || 'image.jpg';

            await NotebookAPI.uploadAttachment(state.token, noteId, blob, filename);
            uploaded++;

            elements.loadingText.textContent = `이미지 업로드 중... (${uploaded}/${imagesToUpload.length})`;
        } catch (error) {
            console.error('Image upload failed:', url, error);
        }
    }
}

/**
 * URL에서 파일명 추출
 */
function extractFilename(url) {
    try {
        const urlObj = new URL(url);
        const pathname = urlObj.pathname;
        const filename = pathname.split('/').pop();
        return filename.split('?')[0] || 'image.jpg';
    } catch {
        return 'image.jpg';
    }
}

/**
 * 페이지 콘텐츠 추출 함수 (Content Script에서 실행)
 * HTML + CSS 원본을 추출하여 서버로 전송 (서버에서 BeautifulSoup으로 정제)
 */
function extractPageContent(includeImages, selectionOnly) {
    const baseUrl = window.location.href;
    const origin = window.location.origin;

    // URL을 절대 경로로 변환하는 헬퍼 함수
    function toAbsoluteUrl(url) {
        if (!url || url.startsWith('data:') || url.startsWith('javascript:') || url.startsWith('#')) {
            return url;
        }
        if (url.startsWith('http://') || url.startsWith('https://')) {
            return url;
        }
        try {
            return new URL(url, baseUrl).href;
        } catch (e) {
            return url;
        }
    }

    // 선택 영역만 저장하는 경우
    if (selectionOnly) {
        const selection = window.getSelection();
        if (selection && selection.rangeCount > 0) {
            const range = selection.getRangeAt(0);
            const container = document.createElement('div');
            container.appendChild(range.cloneContents());

            // 선택 영역 내 링크와 이미지도 절대 경로로 변환
            container.querySelectorAll('a[href]').forEach(a => {
                a.href = toAbsoluteUrl(a.getAttribute('href'));
            });
            container.querySelectorAll('img[src]').forEach(img => {
                img.src = toAbsoluteUrl(img.getAttribute('src'));
            });

            return { html: container.innerHTML, images: [] };
        }
    }

    // body 복제하여 작업 (원본 수정 방지)
    const bodyClone = document.body.cloneNode(true);

    // 모든 링크를 절대 경로로 변환
    bodyClone.querySelectorAll('a[href]').forEach(a => {
        const href = a.getAttribute('href');
        a.setAttribute('href', toAbsoluteUrl(href));
    });

    // 모든 이미지 src를 절대 경로로 변환
    bodyClone.querySelectorAll('img').forEach(img => {
        const src = img.getAttribute('src');
        const dataSrc = img.getAttribute('data-src');
        const dataLazySrc = img.getAttribute('data-lazy-src');

        if (src) img.setAttribute('src', toAbsoluteUrl(src));
        if (dataSrc) img.setAttribute('data-src', toAbsoluteUrl(dataSrc));
        if (dataLazySrc) img.setAttribute('data-lazy-src', toAbsoluteUrl(dataLazySrc));
    });

    // srcset 속성도 처리
    bodyClone.querySelectorAll('[srcset]').forEach(el => {
        const srcset = el.getAttribute('srcset');
        if (srcset) {
            const newSrcset = srcset.split(',').map(part => {
                const [url, descriptor] = part.trim().split(/\s+/);
                return descriptor ? `${toAbsoluteUrl(url)} ${descriptor}` : toAbsoluteUrl(url);
            }).join(', ');
            el.setAttribute('srcset', newSrcset);
        }
    });

    // CSS 추출 (style 태그 + link 태그의 인라인 스타일)
    let cssContent = '';

    // 1. <style> 태그 내용 수집
    document.querySelectorAll('style').forEach(style => {
        cssContent += style.textContent + '\n';
    });

    // 2. 외부 CSS는 URL만 수집 (link 태그)
    const cssLinks = [];
    document.querySelectorAll('link[rel="stylesheet"]').forEach(link => {
        if (link.href) {
            cssLinks.push(link.href);
        }
    });

    // 이미지 URL 추출 (절대 경로로)
    const images = [];
    if (includeImages) {
        bodyClone.querySelectorAll('img').forEach(img => {
            let src = img.getAttribute('src') || img.getAttribute('data-src') || img.getAttribute('data-lazy-src');
            if (src && !src.startsWith('data:') && !images.includes(src)) {
                images.push(src);
            }
        });
    }

    // HTML 원본 + CSS 함께 전송
    const fullHtml = `<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<base href="${origin}">
${cssLinks.map(url => `<link rel="stylesheet" href="${url}">`).join('\n')}
<style>${cssContent}</style>
</head>
<body>
${bodyClone.innerHTML}
</body>
</html>`;

    return { html: fullHtml, images };
}

// Event Listeners
elements.loginForm.addEventListener('submit', handleLogin);
elements.logoutBtn.addEventListener('click', handleLogout);
elements.saveBtn.addEventListener('click', handleSave);

// 초기화
init();
